	function [X, Fmin] = PRAXIS(F, N, H0, PRIN, ILLC, SCBD, KTM, FMIN)
	
	%	PRAXIS returns the minimum of the function F(X, N) of N variables
	%	using the principal axis method. The gradient of the function is not required.
	%	For a description of the algorithm, se chapter 7 of the "Algorithms for 
	%	finding zeros and extrema of functions without calculating derivatives"
	%	by Richard P. Brent: Prentice-Hall, Inc. 1973
	%	This routine is particulary useful, when the evaluation of the 
	%	object function is very time consuming. 

	% 	Toolbox for  DIGITAL FILTERS USING MATLAB 
	
	%	Translated from FORTRAN:	Lars Wanhammar, 2005-11-04
	%	Modified by: 	LW 2010-10-01 
	%					
	% 	Copyright:			by authors - not released for commercial use
	%	Version: 		1
	%	Known bugs:		
	%	Report bugs to:	Wanhammar@gmail.com
	
	%			The parameters are:
	%  Tol		is a tolerance. PRAXIS attempts to return PRAXIS = F(X)
	%			such that if X0 is the true local minimum near X, then
	%			NORM(X-X0) < Tol + SQUAREROOT(eps)*NORM(X).
	%  H0   	is the maximum step size. H0 should be set to about the maximum 
	%			distance from the initial guess to the minimum. (if H0 is set too 
	%			large or too small, the initial rate of convergence may be low.)
	%   N   	(at least two) is the number of variables upon which the function depends.
	%  	PRIN  	Controls the printing of intermediate results.
	%			If PRIN = 0, nothing is printed.
	%			If PRIN = 1, F is printed after every N+1 or N+2 linear minimizations.
	%			Final X is printed, but the intermediate X is printed only if N is at most 4.
	%			if PRIN = 2, the scale factors and the principal values of the 
	%			approximating quadratic form are also printed.
	%			If PRIN = 3, X is also printed after every few linear minimizations.
	%			If PRIN = 4, the principal vector of the approximating quadratic 
	%			form are also printed.
	%   X   	is an array containing on entry a guess of the point of minimum,
	%		 	on return the estimated point of minimum.
	%  F(X,N) 	is the function to be estimated. F should be a real function 
	%			decleared external in the calling program.
	%  FMIN  	is an estimate of the minimum, used only in printing intermediate results. 
	%			The approximating quadratic form is 
	
	%     		Q(X') = F(X,N) + (1/2) * (X'-X)-TRANSPOSE * A * (X'-X)
	
	%			where X is the best estimate of the minimum & A is (V-TRANSPOSE)^-1*D*(V)^-1
	%  V(*,*) 	is the matrix of search directions.  
	%  D(*) 	is the array of second differences.
	%  			If F hads continous second derivatives near X0, A will tend to the 
	%			Hessian of F at X0 as X approaches X0.
	%			It is assumed that on floating-point underflow the result is set to zero.
	
	%			HEURISTIC NUMBERS:
	%	SCBD	If the axes may be badly scaled (which is to be avoided if possible),
	%			set SCBD = 10. Otherwise set SCBD = 1.
	%	ILLC	If the problem is known to be ill-conditioned, set ILLC = True.
	%			Otherwise set ILLC = False.
	%	KTM		KTM is the number of iterations without improvement before the 
	%			algorithm terminates. KTM = 4 is very cautious; usually KTM = 1 is 
	%			satisfactory.
	% ==========================================================
	
	global FX LDT Dmin NF NL V Q0 Q1 QA QB QC QD0 QD1 QF1 Tol Tol1 H X
	global small Vsmall LARGE VLARGE M2 M4 True False

	V = zeros(N); D = zeros(1, N); Y = D; Z = D; Q0 = D; Q1 = D; Tmat = D;
	%		Machine dependent numbers:
	small = eps^2; Vsmall = small^2; LARGE = 1/small; VLARGE = 1/Vsmall;
	M2 = sqrt(eps); M4 = sqrt(M2); True = (1==1); False = (1==0);
	LDFAC = 0.01; if (ILLC == True), LDFAC = 0.1; end
	KT = 0; NL = 0; NF = 1;
	FX = feval(F, X, N); % Evaluate parametric function F using args X and N
	QF1 = FX; Tol1 = small + abs(Tol); T2 = Tol1; Dmin = small; H = H0; 	LDT = H;
	if (H < 100*Tol1), H = 100*Tg; end
	%   The first set of search directions V is the identity matrix
	V = eye(N, N); D(1) = 0; QD0 = 0; Q0 = X; Q1 = X; QA = 0; QB = 0; QC = 0; QD1 = 0;
	if (PRIN > 0) 
		disp(['After ',int2str(NL),' linear searches, the function has been evaluated ',int2str(NF),' times'])
		disp(['The smallest value found is F(x) = ', num2str(FX)])
		if (FX <= FMIN) 
			disp([' Log(F(x) - ', num2str(FMIN), ') is undefined ', num2str(FMIN)])
		else
			disp(['log(F(x) - ' , num2str(FMIN), ') = ', num2str(log10(FX - FMIN))])
		end
		if ~(N > 4 & PRIN <= 2),	X, end
	end
	% ===============================================================
	%   The main loop starts here ......
	Mgoto = 40;
	while Mgoto < 1000
		switch Mgoto
		case 40
			SF = D(1); D(1) = 0; 
			%   Minimize a the first direction V(*,1).
			Value = FX;		%   FX must be passed to MIN by a value. 
			[D(1), S] = Praxis_MIN(F, N, 1, 2, 0, 0, Value, False);	
			if (S <= 0), V(:,1) = -V(:,1); end
			if ~(SF > 0.9*D(1) & 0.9*SF < D(1)), D(2:N) = 0; end
			Mgoto = 70;
			% ============================================================
			%   The inner loop starts here
			K = 2;  
		case 70
			Y = X; SF = FX; if (KT > 0), ILLC = True; end
			Mgoto = 80;
		case 80
			KL = K; DF = 0;
			%   A random step follows (to aviod resolution valleys). 
			%   PRAXIS assumes that rand returns a random number uniformly
			%   distributed in (0,1).
			if (ILLC)
				for i = 1:N
					S = (0.1*LDT + T2*(10^KT))*(rand - 0.5); Z(i) = S;
					X = X + S*V(:,i)';
				end
                FX = feval(F, X, N); % Evaluate parametric function F using args X and N
				NF = NF+1;
			end
			%   Minimize along the "non-conjugate" directions V(*,k),...,V(*,N)
			for K2 = K:N
				SL = FX; Value = FX;
				[D(K2), S] = Praxis_MIN(F, N, K2, 2, D(K2), S, Value, False);	
				if (ILLC), S = D(K2)*((S + Z(K2))^2); else S = SL - FX; end 
				if (DF <= S), DF = S; KL = K2; end 
			end
			if (ILLC | (DF >= abs((100*eps)*FX))), Mgoto = 110; end
			%   If there was not much improvment on the first try, set ILLC = True
			%   and start the inner loop again 
			if (Mgoto ~= 110), ILLC = True; Mgoto = 80; end
		case 110
			if (K == 2 & PRIN > 1)
				disp('The second difference array D(*) is: '), D
			end
			%   Minimize along the "conjugate" directions V(*,1),...,V(*,k-1)
			KM1 = K-1;
			for K2 = 1:KM1
				S = 0; Value = FX;
				[D(K2), S] = Praxis_MIN(F, N, K2, 2, D(K2), S, Value, False);
			end
			F1 = FX; FX = SF; LDS = 0;
			for i = 1:N
				SL = X(i); X(i) = Y(i); SL = SL-Y(i); Y(i) = SL; LDS = LDS + SL^2;
			end
			LDS = sqrt(LDS);
			if (LDS <= small), Mgoto = 160; else Mgoto = 111; end
		case 111
			%  Discard direction V(*,KL).
			%  If no random step was taken, V(*,KL) is the "non-conjugate" 
			%  direction along which the greatest improvment was made
			KLMK = KL - K;
			if (KLMK >= 1)  
				for ii = 1:KLMK
					i = KL-ii; for j = 1:N, V(j,i+1) = V(j,i); end
					D(i+1) = D(i);
				end
			end
			D(K) = 0; V(:, K) = Y(:)/LDS;
			%   Minimize a the new "conjugate" direction V(*,K), which is
			%   the normalized vector: (NEW X) - (0LD X)
			Value = F1;
			[D(K), LDS] = Praxis_MIN(F, N, K, 4, D(K), LDS, Value, True);
			if (LDS > 0), Mgoto = 160; else Mgoto = 112; end
		case 112
			LDS = -LDS; Mgoto = 160; V(:,K) = -V(:,K);
		case 160
			LDT = LDFAC*LDT; if (LDT < LDS), LDT = LDS; end
			if (PRIN > 0)
				disp(['After ',int2str(NL),' linear searches, the function has been evaluated ',int2str(NF),' times'])
				disp(['The smallest value found is F(x) = ',num2str(FX)])
				if (FX <= FMIN) 
					disp([' Log(F(x) - ',num2str(FMIN), ') is undefined ',num2str(FMIN)])
				else 
					disp(['log(F(x) - ' ,num2str(FMIN), ') = ' ,num2str(log10(FX - FMIN))])
				end
				if ~(N > 4 & PRIN <= 2), X, end
			end
			T2 = M2*sqrt(X*X') + Tol1;		
			%   See whether the length of the step taken since starting the
			%   inner loop exceeds half the tolerance 
			if (LDT > 0.5*T2), KT = -1; end
			KT = KT+1; if (KT > KTM), Mgoto = 400; else Mgoto = 170;  end
		case 170
			K = K+1; if (K > N), Mgoto = 171; else Mgoto = 70; end
			%   The inner loop ends here 
			% ===============================================================				
			%   Try quadratic extrapolation in the case we are in a curved valley.
		case 171	%	Looks for the minimum of the F along a curve defined by Q0, Q1, X
			S = FX; FX = QF1; QF1 = S; QD1 = 0;
			for i = 1:N
				S = X(i); L = Q1(i); X(i) = L; Q1(i) = S; QD1 = QD1 + (S-L)^2;
			end
			QD1 = sqrt(QD1); L = QD1; S = 0;
			if ((QD0 <= 0)|(QD1 <= 0)|(NL < 3*N*N))
				FX = QF1; QA = 0; QB = 0; QC = 1;
			else
				Value = QF1;
				[S, L] = Praxis_MIN(F, N, 0, 2, S, L, Value, True);	
				QA = (L*(L- QD1))/(QD0*(QD0 + QD1));
				QB = ((L + QD0)*(QD1 - L))/(QD0*QD1);
				QC = (L*(L + QD0))/(QD1*(QD0 + QD1));
			end
			QD0 = QD1;
			for i = 1:N
				S = Q0(i); Q0(i) = X(i); X(i) = QA*S + QB*X(i) + QC*Q1(i);
			end
			DN = 0; D = D.^(-0.5); DN = max(D);
			if (PRIN > 3)disp('The new directions are: '), V, end
			for j = 1:N, S = D(j)/DN; V(:,j) = S*V(:,j); end
			if (SCBD > 1)	%   Scales the axes to try to reduce the condition number 
				S = VLARGE;
				for i = 1:N
					Z(i) = sqrt(V(i,:)*V(i,:)'); if (Z(i) < M4), Z(i) = M4; end
					if (S > Z(i)), S = Z(i); end
				end
				for i = 1:N
					SL = S/Z(i); Z(i) = 1/SL;
					if (Z(i) > SCBD), SL = 1/SCBD; Z(i) = SCBD; end 
				end
				V(i,:) = SL*V(i,:); 
			end
			%	Calculate a new set of orthogonal directions before repeating the 
			%	main loop.
			%   Find the singular value decomposition of V. 
			%   This gives the principal values and principal directions of the
			%   approximating quadratic form without squaring the condition number.
			[U, Diag, V] = svd(V');	
			for i = 1:N, D(i) = Diag(i,i); end		
			if (SCBD > 1) 		%   Unscale axes
				for i = 1:N, S = Z(i); V(i,:) = S*V(i,:); end
				for i = 1:N
					S = sqrt(V(i,:)*V(i,:)'); 
					D(i) = S*D(i); S = 1/S; V(:,i) = S*V(:,i);
				end
			end
			for i = 1:N
				DNI = DN*D(i);
				if (DNI > LARGE)
					D(i) = Vsmall;
				else
					if (DNI < small), D(i) = VLARGE; else D(i) = 1/DNI^2; end
				end
			end
			Dmin = D(N);
			if (Dmin < small), Dmin = small; end	
			if (M2*D(1) > Dmin), ILLC = True; end
			if (PRIN > 1 & SCBD > 1), disp('The scale factors are:'), Z, end
			if (PRIN > 1)
				disp('The approximating quadratic form has the principal values:'), D
			end
			if (PRIN > 3), disp('and the principal axes: '), V, end
			Mgoto = 40;
			%   The main loop ends here 
			% =========================================================
		case 400
			Fmin = FX;
			if (PRIN > 0), disp(' X is: '), X, Fmin = FX, end
			Mgoto = 10000;		
		end
	end
	return
	
	% ====================================================================
	function [Fout, NF] = Praxis_FEvaluate(F, N, J, L)
	%  Evaluate the function, Praxis_F(X, N), at L in the direction V(., J)
	global FX LDT Dmin NF NL V Q0 Q1 QA QB QC QD0 QD1 QF1 Tol Tol1 H X
	global small Vsmall LARGE VLARGE M2 M4 True False
	Tmat = zeros(1, N);
	if (J == 0) 	%   The search is a parabolic space curve 
		QA = (L*(L - QD1))/(QD0*(QD0 + QD1));
		QB = ((L + QD0)*(QD1 - L))/(QD0*QD1);
		QC = (L*(L + QD0))/(QD1*(QD0 + QD1));
		for i = 1:N, Tmat(i) = QA*Q0(i) + QB*X(i) + QC*Q1(i); end
	else	%   The search is linear 
		Tmat = X + L*V(:,J)';
	end 
	NF = NF + 1;	%   The function evaluation counter NF is incremented
	Fout = feval(F, Tmat, N); % Evaluate parametric function F using args Tmat and N
	return
	
	% ====================================================================
	function [D2, X1] = Praxis_MIN(F, N, J, NITS, D2, X1, Value, FK)	
	%   The subroutine Praxis_MIN minimizes F from X in the direction V(*,J)  
	%   unless J is less than 1, when quadratic search is made in the plane 
	%   defined by Q0, Q1, X. D2 is either zero or an approximation of half F.
	%   On entry, X1 is an estimate of the distance from X to the minimum along
	%   V(*,J) (or, if J = 0, along curve). On return, X1 is the distance found.
	%   If FK = True, then F1 is FLIN(X1). Otherwise X1 and  F1 are ignored
	%   on entry unless final FX is greater than F1.
	%   NITS contols the number of times an attempte will be made to halve 
	%   the intervall.
	global FX LDT Dmin NF NL V Q0 Q1 QA QB QC QD0 QD1 QF1 Tol Tol1 H X
	global small Vsmall LARGE VLARGE M2 M4 True False
	F1 = Value; SF1 = Value; SX1 = X1; K = 0; XM = 0; FM = FX; F0 = FX;
	if (D2 < eps), DZ = True; else DZ = False; end
	S = sqrt(X*X');			%   Find the step size
	if (DZ == True), TEMP = Dmin; else TEMP = D2; end
	T2 = M4*sqrt(abs(FX)/TEMP + S*LDT) + M2*LDT; S = M4*S + Tol1;
	if ((DZ == True) & (T2 > S)), T2 = S; end
	T2 = min(max(T2, small), 0.01*H);
	if ~((FK == False) | (Value > FM)), XM = X1; FM = Value; end 
	if ~((FK == True) & (abs(X1) >= T2))
		if (X1 < 0), X1 = -T2; else X1 = T2; end
		[F1, NF] = Praxis_FEvaluate(F, N, J, X1);
	end 
	if (F1 <= FM), XM = X1; FM = F1; end 
	Iflag = 1;
	while Iflag < 6
		switch Iflag
		case 1  
			if (DZ == True) %   Evaluate Praxis_FEvaluate at another point  
				X2 = -X1; 	%	and estimate the second derivative
				if (F0 >= F1), X2 = 2*X1; end
				[F2, NF] = Praxis_FEvaluate(F, N, J, X2);
				if (F2 <= FM), XM = X2; FM = F2; end 
				D2 = (X2*(F1 - F0)-X1*(F2 - F0))/((X1*X2)*(X1 - X2)); 
			end
			%   Estimate the first derivative at 0 
			D1 = (F1 - F0)/X1 - X1*D2; DZ = True;	
			if (D2 > small)	%   Predict the minimum
				X2 = -0.5*D1/D2;
			else
				X2 = H; if (D1 >= 0), X2 = -X2; end
			end 
			if (abs(X2) > H)
				if (X2 <= 0), X2 = -H; else X2 = H; end
			end 
			Iflag = 4;
		case 4 
			%   Evaluate F at the predicted minimum 
			[F2, NF] = Praxis_FEvaluate(F, N, J, X2);
			if ((K >= NITS) |(F2 <= F0)), Iflag = 10; else Iflag = 5; end
		case 5 	%   No success, so try again 
			K = K + 1;
			if ((F0 < F1) & (X1*X2 > 0))
				Iflag = 1;
			else
				X2 = 0.5*X2; Iflag = 4;
			end 
		end 
	end 
	NL = NL + 1; 	%   Increment the one-dimensional search counter
	if (F2 <= FM), FM = F2; else X2 = XM; end
	%   Get a new estimate of the second derivative 
	if (abs(X2*(X2 - X1)) <= small)
		if (K > 0), D2 = 0; end
	else
		D2 = (X2*(F1-F0) - X1*(FM-F0))/((X1*X2)*(X1 - X2));
	end 
	if (D2 <= small), D2 = small; end
	X1 = X2; FX = FM;
	if (SF1 < FX), FX = SF1; X1 = SX1; end
	if (J ~= 0) 	%   Update X for linear but not parabolic search
		X = X + X1*V(:,J)';
	end
	return % D2 and X1 and X through globals
	% ====================================================================
	
